<?php

/**
 * This software is intended for use with Oxwall Free Community Software http://www.oxwall.org/ and is
 * licensed under The BSD license.

 * ---
 * Copyright (c) 2012, Sergey Kambalin
 * All rights reserved.

 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that the
 * following conditions are met:
 *
 *  - Redistributions of source code must retain the above copyright notice, this list of conditions and
 *  the following disclaimer.
 *
 *  - Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
 *  the following disclaimer in the documentation and/or other materials provided with the distribution.
 *
 *  - Neither the name of the Oxwall Foundation nor the names of its contributors may be used to endorse or promote products
 *  derived from this software without specific prior written permission.

 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/**
 * @author Sergey Kambalin <greyexpert@gmail.com>
 * @package uavatars.classes
 */
class UAVATARS_CLASS_PhotoBridge
{
    /**
     * Class instance
     *
     * @var UAVATARS_CLASS_PhotoBridge
     */
    private static $classInstance;

    /**
     * Returns class instance
     *
     * @return UAVATARS_CLASS_PhotoBridge
     */
    public static function getInstance()
    {
        if ( !isset(self::$classInstance) )
        {
            self::$classInstance = new self();
        }

        return self::$classInstance;
    }

    private $isPluginActive = false;

    /**
     *
     * @var OW_Plugin
     */
    private $plugin;

    private $defaultPhotoAlbumName = 'Profile Pictures';

    private $disabledEvents = array(
        'plugin.photos.add_photo'
    );

    public function __construct()
    {
        $this->isPluginActive = OW::getPluginManager()->isPluginActive('photo');
        $this->plugin = OW::getPluginManager()->getPlugin('uavatars');
    }

    private function triggerEvent( OW_Event $event )
    {
        if ( in_array($event->getName(), $this->disabledEvents) )
        {
            return $event;
        }

        return OW::getEventManager()->trigger($event);
    }

    private function callEvent( $eventName, $params )
    {
        if ( in_array($eventName, $this->disabledEvents) )
        {
            return null;
        }

        return OW::getEventManager()->call($eventName, $params);
    }

    public function isActive()
    {
        return $this->isPluginActive;
    }

    private function getAlbumName()
    {
        $albumName = OW::getLanguage()->text($this->plugin->getKey(), 'default_photo_album_name');

        return empty($albumName) ? $this->defaultPhotoAlbumName : $albumName;
    }

    private function getAlbum( $userId )
    {
        $albumService = PHOTO_BOL_PhotoAlbumService::getInstance();

        $albumName = $this->getAlbumName();
        $album = $albumService->findAlbumByName($albumName, $userId);

        if ( empty($album) )
        {
            $album = new PHOTO_BOL_PhotoAlbum();
            $album->name = $albumName;
            $album->userId = $userId;
            $album->createDatetime = time();

            $albumService->addAlbum($album);
        }

        return $album;
    }

    private function createTemporaryPhoto( $userId, $source )
    {
        $photoService = PHOTO_BOL_PhotoService::getInstance();
        $mainPath = $photoService->getPhotoPath($photo->id);

        $preview = $this->photoTemporaryDao->getTemporaryPhotoPath($tmpPhoto->id, 1);

        $main = $this->photoTemporaryDao->getTemporaryPhotoPath($tmpPhoto->id, 2);
        $original = $this->photoTemporaryDao->getTemporaryPhotoPath($tmpPhoto->id, 3);

        $config = OW::getConfig();
        $width = $config->getValue('photo', 'main_image_width');
        $height = $config->getValue('photo', 'main_image_height');
        $previewWidth = $config->getValue('photo', 'preview_image_width');
        $previewHeight = $config->getValue('photo', 'preview_image_height');

        try {
            $image = new UTIL_Image($source);

            $mainPhoto = $image
                ->resizeImage($width, $height)
                ->saveImage($main);

            if ( (bool) $config->getValue('photo', 'store_fullsize') && $mainPhoto->imageResized() )
            {
                $originalImage = new UTIL_Image($source);
                $res = (int) $config->getValue('photo', 'fullsize_resolution');
                $res = $res ? $res : 1024;
                $originalImage
                    ->resizeImage($res, $res)
                    ->saveImage($original);

                $tmpPhoto->hasFullsize = 1;
                $this->photoTemporaryDao->save($tmpPhoto);
            }

            $mainPhoto
                ->resizeImage($previewWidth, $previewHeight, true)
                ->saveImage($preview);
        }
        catch ( WideImage_Exception $e )
        {
            $this->photoTemporaryDao->deleteById($tmpPhoto->id);
            return false;
        }

        return true;
    }

    /**
     *
     * @return PHOTO_BOL_Photo
     */
    private function upload( $userId, $path, $description )
    {
        $pluginfilesDir = $this->plugin->getPluginFilesDir();
        $source = $pluginfilesDir . uniqid('tmp_') . '.jpg';

        if ( !@OW::getStorage()->copyFileToLocalFS($path, $source) )
        {
            return null;
        }

        $photoService = PHOTO_BOL_PhotoService::getInstance();

        $album = $this->getAlbum($userId);

        $privacy = OW::getEventManager()->call(
            'plugin.privacy.get_privacy',
            array('ownerId' => $userId, 'action' => 'photo_view_album')
        );

        $photo = new PHOTO_BOL_Photo();
        $photo->description = htmlspecialchars($description);
        $photo->albumId = $album->id;
        $photo->addDatetime = time();
        $photo->status = 'approved';
        $photo->hasFullsize = true;
        $photo->privacy = mb_strlen($privacy) ? $privacy : 'everybody';

        $photoService->addPhoto($photo);

        $config = OW::getConfig();
        $width = $config->getValue('photo', 'main_image_width');
        $height = $config->getValue('photo', 'main_image_height');
        $previewWidth = $config->getValue('photo', 'preview_image_width');
        $previewHeight = $config->getValue('photo', 'preview_image_height');

        $tmpMainPath = $pluginfilesDir . 'main_' . $photo->id . '.jpg';
        $tmpPreviewPath = $pluginfilesDir . 'preview_' . $photo->id . '.jpg';
        $tmpOriginalPath = $pluginfilesDir . 'original_' . $photo->id . '.jpg';

        try
        {
            $image = new UTIL_Image($source);

            $mainPhoto = $image
                ->resizeImage($width, $height)
                ->saveImage($tmpMainPath);

            $mainPhoto
                ->resizeImage($previewWidth, $previewHeight, true)
                ->saveImage($tmpPreviewPath);

            if ( $config->getValue('photo', 'store_fullsize') && $mainPhoto->imageResized() )
            {
                $originalImage = new UTIL_Image($source);
                $res = (int) $config->getValue('photo', 'fullsize_resolution');
                $res = $res ? $res : 1024;
                $originalImage
                    ->resizeImage($res, $res)
                    ->saveImage($tmpOriginalPath);

                $photo->hasFullsize = true;
            }
            else
            {
                $photo->hasFullsize = false;
                $photoService->updatePhoto($photo);
            }
        }
        catch ( WideImage_Exception $e )
        {
            @unlink($source);

            return null;
        }

        @unlink($source);

        $storage = OW::getStorage();

        $mainPath = $photoService->getPhotoPath($photo->id);
        $previewPath = $photoService->getPhotoPath($photo->id, 'preview');
        $originalPath = $photoService->getPhotoPath($photo->id, 'original');

        $storage->copyFile($tmpMainPath, $mainPath);
        @unlink($tmpMainPath);
        $storage->copyFile($tmpPreviewPath, $previewPath);
        @unlink($tmpPreviewPath);

        if ( $photo->hasFullsize )
        {
            $storage->copyFile($tmpOriginalPath, $originalPath);
            @unlink($tmpOriginalPath);
        }

        return $photo;
    }

    public function pullPhoto( $photoId )
    {
        if ( !$this->isActive() ) return null;

        $photo = PHOTO_BOL_PhotoService::getInstance()->findPhotoById($photoId);

        if ( empty($photo) )
        {
            return null;
        }

        $source = PHOTO_BOL_PhotoService::getInstance()->getPhotoPath($photoId);

        $pluginfilesDir = $this->plugin->getPluginFilesDir();
        $dist = $pluginfilesDir . uniqid('tmp_') . '.jpg';

        if ( !OW::getStorage()->copyFileToLocalFS($source, $dist) )
        {
            return null;
        }

        return $dist;
    }

    public function isPhotoExists( $photoId )
    {
        return PHOTO_BOL_PhotoService::getInstance()->findPhotoById($photoId) !== null;
    }

    /**
     *
     * @param int $userId
     * @param string $filePath
     * @param string $title
     * @param string $text
     * @return PHOTO_BOL_Photo
     */
    public function addPhoto( $userId, $filePath, $title = null, $text = null )
    {
        if ( !$this->isActive() ) return null;

        $lang = OW::getLanguage();

        $photoService = PHOTO_BOL_PhotoService::getInstance();
        $photoTmpService = PHOTO_BOL_PhotoTemporaryService::getInstance();

        $album = $this->getAlbum($userId);

        $creditsParams = array('pluginKey' => 'photo', 'action' => 'add_photo');

        $credits = $this->callEvent('usercredits.check_balance', $creditsParams);
        if ( $credits === false )
        {
            return;
        }

        $description = empty($title) ? $text : $title;
        $description = empty($description) ? null : $description;

        $photo = $this->upload($userId, $filePath, $description);

        if ( empty($photo) )
        {
            return;
        }

        $this->callEvent('usercredits.track_action', $creditsParams);

        $event = new OW_Event('plugin.photos.add_photo', array(
            array('addTimestamp' => time(), 'photoId' => $photo->id)
        ));

        $this->triggerEvent($event);

        return $photo;
    }

    public function initPhotoFloatBox()
    {
        OW::getDocument()->addScript(OW::getPluginManager()->getPlugin('base')->getStaticJsUrl() . 'jquery.bbq.min.js');
        OW::getDocument()->addScript(OW::getPluginManager()->getPlugin('photo')->getStaticJsUrl() . 'photo.js');

        OW::getLanguage()->addKeyForJs('photo', 'tb_edit_photo');
        OW::getLanguage()->addKeyForJs('photo', 'confirm_delete');
        OW::getLanguage()->addKeyForJs('photo', 'mark_featured');
        OW::getLanguage()->addKeyForJs('photo', 'remove_from_featured');

        $objParams = array(
            'ajaxResponder' => OW::getRouter()->urlFor('PHOTO_CTRL_Photo', 'ajaxResponder'),
            'fbResponder' => OW::getRouter()->urlForRoute('photo.floatbox')
        );

        $script = UTIL_JsGenerator::composeJsString('UAVATARS.setup({$settings});', array(
            'settings' => $objParams
        ));

        OW::getDocument()->addOnloadScript($script);
    }
}