<?php

class THEMEMANAGEROF_BOL_Service
{
	/**
     * Singleton instance.
     *
     * @var THEMEMANAGEROF_BOL_Service
     */
	private static $classInstance;

	/**
	 * @var Array
	 */
	private $src = array();

	/**
     * @var OW_User
     */
    private $user;

    /**
     * @var BOL_UserService
     */
    private $userService;

    /**
     * @var BOL_AvatarService
     */
    private $avatarService;

    /**
     * Returns an instance of class (singleton pattern implementation).
     *
     * @return THEMEMANAGEROF_BOL_Service
     */
    public static function getInstance()
    {
        if ( self::$classInstance === null )
        {
            self::$classInstance = new self();
        }

        return self::$classInstance;
    }

    /**
     * Constructor
     */
    private function __construct()
    {
    	$this->user = OW::getUser();
		$this->userService = BOL_UserService::getInstance();
		$this->avatarService = BOL_AvatarService::getInstance();

		// Construct JS
		$this->addScript('window.OF_ThemeManager = {};');
		$this->addScript('window.OF_ThemeManager.User = ' . json_encode($this->getUser()) . ';');
		$this->addScript('window.OF_ThemeManager.Stats = ' . json_encode($this->getStats()) . ';');
		$this->addScript('window.OF_ThemeManager.imagesUrl = "' . $this->getThemeStaticDir() . 'images/";');
		$this->addScript('window.OF_ThemeManager.baseUrl = "' . OW_URL_HOME . '"');
    }

    /**
     * Bind Events
     */
    public function bindEvents()
    {
    	OW::getEventManager()->bind(OW_EventManager::ON_BEFORE_DOCUMENT_RENDER, array($this, 'renderJS'));
    }

    /**
     * Check if current route is on admin
     * @return boolean
     */
	public function isOnAdminPage()
	{
		$route = explode("/", OW::getRouter()->getUri());
		return $route[0] === "admin";
	}

	/**
	 * @return OW_User
	 */
	public function getOWUser()
	{
		return $this->user;
	}

	/**
	 * @return string
	 */
	public function getSelectedTheme()
	{
		return OW::getConfig()->getValue('base', 'selectedTheme');
	}

	/**
	 * @return string
	 */
	public function getThemeDir()
	{
		return OW_DIR_THEME . self::getSelectedTheme() . DS;
	}

	public function getThemeStaticDir()
	{
		return OW_URL_STATIC_THEMES . self::getSelectedTheme() . "/";
	}

	/**
	 * @return Object
	 */
	public function getUser()
	{
		$owuser = $this->user->getUserObject();
		$userId = $this->user->getId();

		return (object) array(
			'url' => $this->userService->getUserUrl($userId),
			'avatar' => $this->avatarService->getAvatarUrl($userId),
			'username' => isset($owuser->username) ? $owuser->username : null,
			'displayname' => $this->userService->getDisplayName($userId),
			'admin' => $this->user->isAdmin()
		);
	}

	/**
	 * @return Object
	 */
	public function getStats()
	{
		$ForumService = FORUM_BOL_ForumService::getInstance();
		$EventService = EVENT_BOL_EventService::getInstance();
		$GroupService = GROUPS_BOL_Service::getInstance();
		$BlogsService = PostService::getInstance();
		$userId = $this->user->getId();
		$FriendService = FRIENDS_BOL_Service::getInstance();
		$LinkService  = LinkService::getInstance();
		$VideoService = VIDEO_BOL_ClipService::getInstance();
		$PhotoService = PHOTO_BOL_PhotoService::getInstance();
		$PhotoAlbumService = PHOTO_BOL_PhotoAlbumService::getInstance();

		return (object) array(
			'userGroups' => $GroupService->findUserGroupListCount($userId),
			'userBlogPosts' => $BlogsService->countUserPost($userId),
			'userFriends' => $FriendService->countFriends($userId),
			'userLinks' => $LinkService->countUserLinks($userId),
			'userVideos' => $VideoService->findUserClipsCount($userId),
			'userEvents' => $EventService->findUsersEventsCount($userId),
			'userPhotos' => $PhotoService->countUserPhotos($userId),
			'userAlbums' => $PhotoAlbumService->countUserAlbums($userId),
			'userForums' => count($ForumService->getUserTopicList($userId)),
			'registeredUsers' => $this->userService->count(),
			'forumTopics' => $ForumService->countAllTopics(),
			'groupsPopular' => $GroupService->findGroupListCount(GROUPS_BOL_Service::LIST_MOST_POPULAR),
			'groupsLatest' => $GroupService->findGroupListCount(GROUPS_BOL_Service::LIST_LATEST),
			'albums' => $PhotoAlbumService->countAlbums(),
			'online' => $this->userService->countOnline(),
			'blogPosts' => $BlogsService->countPosts(),
			'events' => $EventService->findPublicEventsCount(),
			'links' => $LinkService->countLinks(),
			'videos' => $VideoService->findClipsCount('latest')
		);
	}

	/**
	 * Add Script
	 *
	 * @param $script string
	 * @param $vars array
	 * @return null
	 */
	public function addScript( $script, $vars = null )
	{
		if ($script === null) return;
		$this->src[] = $this->parseString( $script, $vars );
	}

	/**
	 * Add JS Script
	 *
	 * @param $file string
	 * @param $vars array
	 * @return null
	 */
	public function addJsFile( $file, $vars = null )
	{
		$script = $this->getJSFileContent( $file );
		if ($script === null || trim($script) === "") return;
		$this->addScript( $script, $vars );
	}

	/**
	 * Get JS File Content
	 *
	 * @param $file string
	 * @return string
	 */
	public function getJSFileContent( $file )
	{
		$jsf = $this->getThemeDir() . 'images' . DS . 'js' . DS . $file;
		if ( file_exists($jsf) ) return file_get_contents( $jsf );
		return null;
	}

	/**
	 * Parser
	 * 
	 * @param $s string
	 * @param $vars array
	 * @return string
	 */
	public function parseString( $s, $vars = null)
	{
		if ($vars !== null)
		{
			foreach ($vars as $key => $value) 
			{
				$s = str_replace("%{{" . $key . "}}", $value, $s);
			}
		}
		return $s;
	}

	/**
	 * renderJS
	 */
	public function renderJS()
	{
		// Execute `theme.php`
		$themephp = $this->getThemeDir() . "theme.php";

		if ( file_exists( $themephp ) ) 
		{
			include_once( $themephp );
			run($this);
		}

		if ( count($this->src) > 0 )
		{
			OW::getDocument()->addScriptDeclaration(implode("\r\n", $this->src));
		}
	}
}